/**
 * \file: exchnd_backend_dlt.h
 *
 * Implementation of an exception handler backend that dumps exception handler
 * messages to DLT.
 *
 * \component: exchndd
 *
 * \author: Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <dlt/dlt.h>
#include <dlfcn.h>

#include "exchnd_backend.h"
#include "exchnd_interface.h"

#ifndef _DLT_PACKAGE_MAJOR_VERSION
#   define _DLT_PACKAGE_MAJOR_VERSION 1
#   define _DLT_PACKAGE_MINOR_VERSION 0
#endif

void *libdlt_handle;

int (*dlt_register_app_p)(const char *, const char *);
int (*dlt_unregister_app_p)(void);
int (*dlt_register_context_p)(DltContext *, const char *, const char *);
int (*dlt_unregister_context_p)(DltContext *);
int (*dlt_user_log_write_start_p)(DltContext *,
                                  DltContextData *,
                                  DltLogLevelType);
int (*dlt_user_log_write_string_p)(DltContextData *, const char *);
int (*dlt_user_log_write_finish_p)(DltContextData *);

/*
 * \func dlt_load
 *
 * Load dlt libraries and get needed functions symbols.
 *
 */
static int dlt_load(void)
{
    libdlt_handle = dlopen("libdlt.so", RTLD_NOW | RTLD_GLOBAL);

    if (!libdlt_handle) {
        exchnd_print_error("Not able to load DLT libraries. (%s)",
                           dlerror());
        return 0;
    }

    dlt_register_app_p = dlsym(libdlt_handle, "dlt_register_app");

    if (dlt_register_app_p == NULL) {
        exchnd_print_error("Failed to load libdlt: %s\n", dlerror());
        dlclose(libdlt_handle);
        return 0;
    }

    dlt_unregister_app_p = dlsym(libdlt_handle, "dlt_unregister_app");

    if (dlt_unregister_app_p == NULL) {
        exchnd_print_error("Failed to load libdlt: %s\n", dlerror());
        dlclose(libdlt_handle);
        return 0;
    }

    dlt_register_context_p = dlsym(libdlt_handle, "dlt_register_context");

    if (dlt_register_context_p == NULL) {
        exchnd_print_error("Failed to load libdlt: %s\n", dlerror());
        dlclose(libdlt_handle);
        return 0;
    }

    dlt_unregister_context_p = dlsym(libdlt_handle, "dlt_unregister_context");

    if (dlt_unregister_context_p == NULL) {
        exchnd_print_error("Failed to load libdlt: %s\n", dlerror());
        dlclose(libdlt_handle);
        return 0;
    }

    dlt_user_log_write_start_p = dlsym(libdlt_handle,
                                       "dlt_user_log_write_start");

    if (dlt_user_log_write_start_p == NULL) {
        exchnd_print_error("Failed to load libdlt: %s\n", dlerror());
        dlclose(libdlt_handle);
        return 0;
    }

    dlt_user_log_write_string_p = dlsym(libdlt_handle,
                                        "dlt_user_log_write_string");

    if (dlt_user_log_write_string_p == NULL) {
        exchnd_print_error("Failed to load libdlt: %s\n", dlerror());
        dlclose(libdlt_handle);
        return 0;
    }

    dlt_user_log_write_finish_p = dlsym(libdlt_handle,
                                        "dlt_user_log_write_finish");

    if (dlt_user_log_write_finish_p == NULL) {
        exchnd_print_error("Failed to load libdlt: %s\n", dlerror());
        dlclose(libdlt_handle);
        return 0;
    }

    return 1;
}

/*
 * \func dlt_unload
 *
 * Unload dlt libraries
 */
static void dlt_unload(void)
{
    dlclose(libdlt_handle);
}

/* PRQA: Lint Message 19: DLT macros generate this lint findings*/
/*lint -save -e19 */
DLT_DECLARE_CONTEXT(exchnd_dlt);
/*lint -restore */

typedef struct ExchndBackendDLT {
    ExchndBackend_t base;
} ExchndBackendDLT_t;

/*
 * \func retrieve_msg
 *
 * Retrieve message from msg_list and store it in the backend.
 *
 * \param backend Pointer to the original backend.
 */
static void *retrieve_msg(void *backend)
{
    ExchndBackendDLT_t *DLT_backend = (ExchndBackendDLT_t *)backend;
    struct msg_list *current = mlist;
    char *msg = NULL;
    pthread_mutex_t *msg_lck = &mlist->msg_lck;

    pthread_mutex_lock(msg_lck);
    prctl(PR_SET_NAME, "be_dlt", NULL, NULL, NULL);

#if (DEFAULT_BACKEND & DLT_BACKEND)
    /* Default backend ready to receive data. */
    pthread_cond_broadcast(&msg_handled);
#endif

    pthread_cond_wait(&new_msg, msg_lck);

    while (!(action & EXCHND_TERMINATE)) {
        current = mlist->next;

        while (current != mlist) {
            if (current->read_flag & DLT_BACKEND) {
                current = current->next;
                continue;
            }

            msg = current->msg;
            /* No need to keep the lock during write operation */
            pthread_mutex_unlock(msg_lck);
            DLT_backend->base.store(NULL, 0, msg);

            /* Get lock again to update message structure */
            pthread_mutex_lock(msg_lck);
            current->read_flag |= DLT_BACKEND;
            current = current->next;
        }

#if (DEFAULT_BACKEND & DLT_BACKEND)
        /* List complete for default backend */
        pthread_cond_broadcast(&msg_handled);
#endif

        pthread_cond_wait(&new_msg, msg_lck);
    }

    pthread_mutex_unlock(msg_lck);

    return NULL;
}

/*
 * \func
 *
 * Write a message to the DLT viewer
 *
 * \param buf Data to be written to the DLT viewer
 *
 * \return DLT_RETURN_OK on success or error status on failure
 */
static int send_dlt_msg(char *buf)
{
    DltContextData log;
    int err = DLT_RETURN_ERROR;

    if (dlt_user_log_write_start_p(&exchnd_dlt, &log,
                                   DLT_LOG_INFO) > DLT_RETURN_OK) {
        err = dlt_user_log_write_string_p(&log, (const char *)buf);

        if (err >= DLT_RETURN_OK)
            (void)dlt_user_log_write_finish_p(&log);
        else
            exchnd_print_error("Failed to send message to DLT logger");
    } else {
        exchnd_print_error("Failed to start DLT logger");
    }

    return err;
}

/*
 * \func
 *
 * Write a message to the DLT viewer splitting it line by line
 *
 * \param msg Data to be written to the DLT viewer
 *
 * \return DLT_RETURN_OK on success or error status on failure
 */
static int split_send_dlt_msg(char *msg)
{
    int err = DLT_RETURN_OK;
    char *ch = strchr(msg, '\n');

    /* Log the exception message to the viewer. */
    while (ch != NULL && err == DLT_RETURN_OK) {
        *ch = '\0';
        err = send_dlt_msg(msg);

        msg = ch + 1;
        ch = strchr(msg, '\n');
    }

    return err;
}

/*
 * \func store
 *
 * Write a message to the DLT Viewer
 *
 * \param backend Pointer to backend structure (must be DLT!)
 * \param len Length of the data
 * \param msg Data to be written to the DLT viewer
 *
 * \return 0 if successful, error code otherwise
 */
static unsigned int store(struct ExchndBackend *backend,
                          int len, char *msg)
{
    unsigned int ret = 0;
    /* let satisfy lint. */
    (void)backend;
    (void)len;
    char *buffer = strdup(msg);

    if (buffer != NULL) {
        ret = split_send_dlt_msg(buffer);
        free(buffer);
    } else {
        exchnd_print_error("Insufficient memory was available to "
                           "create DLT message copy");
    }

    return ret;
}

/*
 * \func destroy
 *
 * Destroy the given DLT backend structure and its contents. The function
 * deregisters the context and the application from DLT
 *
 * \param backend Pointer to backend structure (must be DLT!)
 */
static void destroy(struct ExchndBackend *backend)
{
    ExchndBackendDLT_t *DLT_backend = (ExchndBackendDLT_t *)backend;

    dlt_unregister_context_p(&exchnd_dlt);
    dlt_unregister_app_p();

    /* Wait until any pending message is processed. */
    pthread_join(DLT_backend->base.BackendThread, NULL);

    dlt_unload();

    free(DLT_backend);
}

/*
 * \func exchnd_backend_create_dlt
 *
 * Create the DLT backend. The function registers the context and the
 * application with DLT.
 *
 * \param next Pointer to linked list of backends
 * \param filename Filename of the file this backend writes to
 *
 * \return pointer to new backend or NULL memory allocation failed
 */
ExchndBackend_t *exchnd_backend_create_dlt(ExchndBackend_t *next)
{
    ExchndBackendDLT_t *DLT_backend = calloc(1, sizeof(ExchndBackendDLT_t));
    pthread_attr_t attr;

    if (DLT_backend != NULL) {
        /* setup the exception handler backend interface */
        DLT_backend->base.store = store;
        DLT_backend->base.destroy = destroy;
        DLT_backend->base.next = next;

        if (!dlt_load()) {
            free(DLT_backend);
            return next;
        }

        dlt_register_app_p("EXCHND", "exception handler daemon");
        dlt_register_context_p(&exchnd_dlt,
                               "EXCHND",
                               "exception handler daemon");

        pthread_attr_init(&attr);
        pthread_attr_setstacksize(&attr, PTHREAD_STACK_MIN);

        pthread_create(&DLT_backend->base.BackendThread,
                       &attr,
                       retrieve_msg,
                       DLT_backend);

        pthread_attr_destroy(&attr);

        /* We assume base to be the first element. */
        return (ExchndBackend_t *)DLT_backend;
    } else {
        return next;
    }
}